<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Transaction;
use App\Models\User;
use Illuminate\Http\Request;
use Carbon\Carbon;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;

class TransactionController extends Controller
{
    public function __construct()
    {
        $this->checkAdminRole();
    }

    private function checkAdminRole(): void
    {
        if (!Auth::check() || Auth::user()->role !== 'admin') {
            abort(403, 'Unauthorized. Admin access required.');
        }
    }
    
    public function index()
    {
        $transactions = Transaction::with(['user', 'items.product'])
            ->when(request('search'), function ($query) {
                $query->where('id', 'like', '%' . request('search') . '%')
                    ->orWhereHas('user', function ($q) {
                        $q->where('name', 'like', '%' . request('search') . '%');
                    });
            })
            ->when(request('user_id'), function ($query) {
                $query->where('user_id', request('user_id'));
            })
            ->when(request('date_from'), function ($query) {
                $query->whereDate('transaction_date', '>=', request('date_from'));
            })
            ->when(request('date_to'), function ($query) {
                $query->whereDate('transaction_date', '<=', request('date_to'));
            })
            ->when(request('amount_min'), function ($query) {
                $query->where('total_amount', '>=', request('amount_min'));
            })
            ->when(request('amount_max'), function ($query) {
                $query->where('total_amount', '<=', request('amount_max'));
            })
            ->orderBy('transaction_date', 'desc')
            ->paginate(15);

        $users = User::where('role', 'kasir')->orderBy('name')->get();

        // Statistics
        $totalRevenue = $transactions->sum('total_amount');
        $totalTransactions = $transactions->total();

        return view('admin.transactions.index', compact('transactions', 'users', 'totalRevenue', 'totalTransactions'));
    }

    /**
     * Display the specified transaction
     */
    public function show(Transaction $transaction)
    {
        $transaction->load(['user', 'items.product.category']);

        return view('admin.transactions.show', compact('transaction'));
    }

    /**
     * Generate sales report
     */
    public function report()
    {
        $dateFrom = request('date_from', now()->startOfMonth()->format('Y-m-d'));
        $dateTo = request('date_to', now()->format('Y-m-d'));

        // Daily sales
        $dailySales = Transaction::selectRaw('DATE(transaction_date) as date, COUNT(*) as count, SUM(total_amount) as revenue')
            ->whereBetween('transaction_date', [$dateFrom, $dateTo])
            ->groupBy('date')
            ->orderBy('date')
            ->get();

        // Sales by user
        $salesByUser = Transaction::with('user')
            ->selectRaw('user_id, COUNT(*) as count, SUM(total_amount) as revenue')
            ->whereBetween('transaction_date', [$dateFrom, $dateTo])
            ->groupBy('user_id')
            ->orderBy('revenue', 'desc')
            ->get();

        // Top selling products
        $topProducts = DB::table('transaction_items')
            ->join('transactions', 'transaction_items.transaction_id', '=', 'transactions.id')
            ->join('products', 'transaction_items.product_id', '=', 'products.id')
            ->selectRaw('products.name, SUM(transaction_items.quantity) as total_quantity, SUM(transaction_items.subtotal) as total_revenue')
            ->whereBetween('transactions.transaction_date', [$dateFrom, $dateTo])
            ->groupBy('products.id', 'products.name')
            ->orderBy('total_quantity', 'desc')
            ->limit(10)
            ->get();

        // Summary statistics
        $summary = [
            'total_transactions' => Transaction::whereBetween('transaction_date', [$dateFrom, $dateTo])->count(),
            'total_revenue' => Transaction::whereBetween('transaction_date', [$dateFrom, $dateTo])->sum('total_amount'),
            'average_transaction' => Transaction::whereBetween('transaction_date', [$dateFrom, $dateTo])->avg('total_amount'),
            'total_items_sold' => DB::table('transaction_items')
                ->join('transactions', 'transaction_items.transaction_id', '=', 'transactions.id')
                ->whereBetween('transactions.transaction_date', [$dateFrom, $dateTo])
                ->sum('transaction_items.quantity'),
        ];

        return view('admin.transactions.report', compact(
            'dailySales',
            'salesByUser',
            'topProducts',
            'summary',
            'dateFrom',
            'dateTo'
        ));
    }

    /**
     * Export transactions to CSV
     */
    public function export()
    {
        $dateFrom = request('date_from', now()->startOfMonth()->format('Y-m-d'));
        $dateTo = request('date_to', now()->format('Y-m-d'));

        $transactions = Transaction::with(['user', 'items.product'])
            ->whereBetween('transaction_date', [$dateFrom, $dateTo])
            ->orderBy('transaction_date', 'desc')
            ->get();

        $filename = 'transactions_' . $dateFrom . '_to_' . $dateTo . '.csv';

        $headers = [
            'Content-Type' => 'text/csv',
            'Content-Disposition' => 'attachment; filename="' . $filename . '"',
        ];

        $callback = function () use ($transactions) {
            $file = fopen('php://output', 'w');

            // CSV Headers
            fputcsv($file, [
                'ID Transaksi',
                'Tanggal',
                'Kasir',
                'Total Items',
                'Total Amount',
                'Payment Amount',
                'Change Amount',
                'Detail Items'
            ]);

            // CSV Data
            foreach ($transactions as $transaction) {
                $itemsDetail = $transaction->items->map(function ($item) {
                    return $item->product->name . ' (x' . $item->quantity . ')';
                })->implode('; ');

                fputcsv($file, [
                    $transaction->id,
                    $transaction->transaction_date->format('Y-m-d H:i:s'),
                    $transaction->user->name,
                    $transaction->items->sum('quantity'),
                    $transaction->total_amount,
                    $transaction->payment_amount,
                    $transaction->change_amount,
                    $itemsDetail,
                ]);
            }

            fclose($file);
        };

        return response()->stream($callback, 200, $headers);
    }

    /**
     * Delete transaction (with confirmation)
     */
    public function destroy(Transaction $transaction)
    {
        // Only allow deletion if transaction is from today (for security)
        if (!$transaction->transaction_date->isToday()) {
            return back()->with('error', 'Hanya transaksi hari ini yang dapat dihapus!');
        }

        // Restore product stock
        foreach ($transaction->items as $item) {
            $item->product->increment('stock', $item->quantity);
        }

        $transactionId = $transaction->id;
        $transaction->delete();

        return redirect()->route('admin.transactions.index')
            ->with('success', "Transaksi #$transactionId berhasil dihapus dan stok produk telah dikembalikan!");
    }

    /**
     * Dashboard analytics data
     */
    public function analytics()
    {
        // Revenue trend (last 30 days)
        $revenueTrend = Transaction::selectRaw('DATE(transaction_date) as date, SUM(total_amount) as revenue')
            ->where('transaction_date', '>=', now()->subDays(30))
            ->groupBy('date')
            ->orderBy('date')
            ->get();

        // Hourly sales pattern
        $hourlySales = Transaction::selectRaw('HOUR(transaction_date) as hour, COUNT(*) as count, SUM(total_amount) as revenue')
            ->whereDate('transaction_date', today())
            ->groupBy('hour')
            ->orderBy('hour')
            ->get();

        // Category performance
        $categoryPerformance = DB::table('transaction_items')
            ->join('transactions', 'transaction_items.transaction_id', '=', 'transactions.id')
            ->join('products', 'transaction_items.product_id', '=', 'products.id')
            ->join('categories', 'products.category_id', '=', 'categories.id')
            ->selectRaw('categories.name, SUM(transaction_items.quantity) as total_quantity, SUM(transaction_items.subtotal) as total_revenue')
            ->whereDate('transactions.transaction_date', '>=', now()->startOfMonth())
            ->groupBy('categories.id', 'categories.name')
            ->orderBy('total_revenue', 'desc')
            ->get();

        return view('admin.transactions.analytics', compact(
            'revenueTrend',
            'hourlySales',
            'categoryPerformance'
        ));
    }

    /**
     * Print receipt
     */
    public function printReceipt(Transaction $transaction)
    {
        $transaction->load(['user', 'items.product']);

        return view('admin.transactions.receipt', compact('transaction'));
    }
}
