<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Product;
use App\Models\Category;
use App\Models\Transaction;
use App\Models\User;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;

class DashboardController extends Controller
{
    public function __construct()
    {
        $this->checkAdminRole();
    }

    private function checkAdminRole(): void
    {
        if (!Auth::check() || Auth::user()->role !== 'admin') {
            abort(403, 'Unauthorized. Admin access required.');
        }
    }

    public function index()
    {
        $totalProducts = Product::count();
        $totalCategories = Category::count();
        $totalTransactions = Transaction::count();
        $totalRevenue = Transaction::sum('total_amount');
        $lowStockProducts = Product::where('stock', '<=', 10)->count();
        
        $recentTransactions = Transaction::with('user')
            ->orderBy('created_at', 'desc')
            ->limit(5)
            ->get();
            
        // Data untuk chart (opsional) - Fixed query
        $monthlyRevenue = [];
        for ($i = 1; $i <= 12; $i++) {
            $revenue = Transaction::whereYear('transaction_date', now()->year)
                ->whereMonth('transaction_date', $i)
                ->sum('total_amount') ?? 0;
            $monthlyRevenue[$i] = $revenue;
        }
            
        // Fixed top products query using proper join
        $topProducts = Product::select('products.*', DB::raw('COALESCE(SUM(transaction_items.quantity), 0) as transaction_items_count'))
            ->leftJoin('transaction_items', 'products.id', '=', 'transaction_items.product_id')
            ->groupBy('products.id')
            ->orderBy('transaction_items_count', 'desc')
            ->limit(5)
            ->get();
            
        return view('admin.Dashboard', compact(
            'totalProducts', 
            'totalCategories', 
            'totalTransactions', 
            'totalRevenue',
            'lowStockProducts',
            'recentTransactions',
            'monthlyRevenue',
            'topProducts'
        ));
    }
}